#include <avr/io.h>
#include <avr/signal.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define SCANFREQ	0xD9//0xD9
#define SCANPRES	0x03

#define OFF 0
#define ENTER 1
#define MIN 2
#define STD 3

#define UP 1
#define DOWN 2

#define BUFSIZE 64

volatile unsigned char screenBuf[5]={0,0,0,0,0};
volatile unsigned char screenStr[64]={0,0,0,0,0,0};
volatile unsigned char timeStr[64]={0,0,0,0,0,0};
volatile unsigned char menuStr[16]={0,0,0,0,0,0};

volatile unsigned char menu=OFF;

volatile unsigned char strpos = 0, end=0;
volatile int scrollpos = 0;
volatile unsigned int poscnt = 0;

volatile unsigned char seconds=0;
volatile unsigned char minutes=00;
volatile unsigned char hours = 14;

// define the fonts

static unsigned char __attribute__ ((progmem))  font[] = {
						5,5,7,5,7,
						7,5,3,5,7,
						6,1,1,1,6,
						3,5,5,5,3,
						7,1,3,1,7,
						1,1,3,1,7,
						7,5,7,1,7,
						5,5,7,5,5,
						2,2,2,2,2,
						1,2,2,0,2,	// <---
						5,3,1,3,5,
						7,1,1,1,1,
						5,5,5,7,5,
						5,5,5,3,0,	// <---
						7,5,5,5,7,
						1,1,7,5,7,
						4,7,5,5,7,	// <---
						5,3,7,5,7,
						7,4,7,1,7,
						2,2,2,2,7,
						7,5,5,5,5,
						2,5,5,5,5,
						5,7,5,5,5,
						5,5,2,5,5,
						7,4,7,5,5,
						7,1,2,4,7};
						
static unsigned char __attribute__ ((progmem)) digits[] = {
						7,5,5,5,7,
						2,2,2,3,2,
						7,1,7,4,7,
						7,4,7,4,7,
						4,4,7,5,5,
						7,4,7,1,7,
						7,5,7,1,7,
						4,4,4,4,7,
						7,5,7,5,7,
						7,4,7,5,7};
						
static unsigned char __attribute__ ((progmem)) special[] = {
						0,0,0,0,0, 	//space
						2,0,2,2,2, 	// !
						0,0,0,5,5,	// "
						5,7,5,7,7,	// #
						0,0,0,0,0,	// $
						0,0,0,0,0,	// %
						0,0,0,0,0,	// &
						0,0,0,4,4,	// '
						2,1,1,1,2,	// (
						2,4,4,4,2,	// )
						0,0,2,0,0,	// *
						0,2,7,2,0,	// +
						1,2,0,0,0,	// ,
						0,0,7,0,0,	// -
						2,0,0,0,0,	// .
						1,1,2,4,4,	// /
						0,2,0,2,0,	// : Offset 58
						1,2,0,2,0,	// ;
						4,2,1,2,4,	// <
						0,7,0,7,0,	// =
						1,2,4,2,1	// >
					};	
					
// sets the matrix rows					
void setRows(unsigned char num) {
	PORTC = num&0x07;
	PORTD = (num<<1)&0xF0;
}

// writes a char into the screen buffer
void writeChar(unsigned char ch, int pos) {
	unsigned int i,k=0;
	void *charset=digits;
	unsigned char offset='0';
	//pos*=4;
	
	if (ch >= '0' && ch <= '9') {
		charset = digits;
		offset = '0';
	} else if (ch>='a' && ch<='z'){
		charset = font;
		offset = 'a';
	} else if (ch>=32 && ch <=47) {
		charset = special;
		offset = 32;
	} else if (ch>=':') {
		charset = special;
		offset = ':'-16;		
	}
	
	if (pos < 0) {
		k = -pos;
		pos = 0;
				
	}
	
	for (i=0; i<5; i++) {
		if (ch == 255)
			screenBuf[i]=0xFF;
		else
			screenBuf[i] |= ((( PRG_RDB(&charset[i+(ch-offset)*5]))>>k)<<pos)&0xFF;
	}		

}


// clears the screen buffer

void clearBuf(void) {
	unsigned int i;
	
	for (i=0; i<5; i++) {
		screenBuf[i] = 0x00;
	}		
}

// simple delay loop
void delay(unsigned int time) {
	unsigned int k;
	
	for (k=0; k<time*10; k++) {
		asm volatile ("nop");
	}
}

// copy one string to another
void mstrcpy(unsigned char *d, unsigned char *s) {
	unsigned char i;
	
	for (i=0; s[i]!='\0'; i++)
		d[i] = s[i];

	d[i]='\0';
}

// copy one string from PROGMEM to other string in RAM at offset
void strcatP(unsigned char *d, void *s, unsigned char off) {
	unsigned char i;
	unsigned char ch;
	
	for (i=off; (ch = PRG_RDB(&s[i-off]))!='\0'; i++)
		d[i] = ch;
	d[i] = '\0';
}

// Screen Update ISR
// Do scrolling & refresh screen
INTERRUPT(SIG_OVERFLOW0)
{
	cli();
	unsigned char j,k;
	
	if (menu == OFF) {
		mstrcpy(screenStr, timeStr);
	} else {
		mstrcpy(screenStr, menuStr);
		scrollpos = 0;
		strpos = 0;
		poscnt = 0;
	}
	
	if (poscnt == 0) {
		clearBuf();
		scrollpos = 0;
		writeChar(screenStr[strpos],scrollpos);
		writeChar(screenStr[strpos+1],scrollpos+4);
	}
	
	if (poscnt >= 15) {
		clearBuf();
		scrollpos -= 1;
		
		if (scrollpos == -3) {
			strpos+=1;
			scrollpos = 1;
			
			if (end == 1) {
				end = 2;
			} else if (end ==2) {
				end = 3;
			} else if (end == 3) {
				end = 0;
				strpos = 0;
				scrollpos=7;
			}
		}
		
		if (end<3) writeChar(screenStr[strpos],scrollpos);
		if (end<2) writeChar(screenStr[strpos+1],scrollpos+4);
		if (end<1 && scrollpos == -2) {
			if (screenStr[strpos+2]=='\0') {
				end = 1;	
			} else {
				writeChar(screenStr[strpos+2],scrollpos+8);
			}
		}
			
		poscnt = 0;
	}
	
	for (k=0; k<2; k++) {
		for (j=0; j<5; j++) {
			setRows(screenBuf[j]);			
			PORTB = (1<<j);
			delay(27);			
			PORTB = 0;
		}	
	}
	
	TCNT0 = SCANFREQ;
	
	if (menu==OFF)	
		poscnt++;
	
	sei();
}

// Time Counter - 32,768kHz crystal
INTERRUPT(SIG_OVERFLOW2)
{	
	cli();
	//loop++;
	
	//if (loop == 8) {
		
		seconds++;
		//loop = 0;
		//sprintf (screenStr,":;<>=",hours,minutes,seconds);

/*		timeStr[5]=' ';
		timeStr[6]=' ';
		timeStr[7]='2';
		timeStr[8]='3';
		timeStr[9]='.';
		timeStr[10]='4';
		timeStr[11]=' ';
		timeStr[12]='c';		
		timeStr[13]='\0';*/
		/*screenStr[0]=seconds/10+'0';
		screenStr[1]=seconds%10+'0';	
		screenStr[2]='\0';		*/
	//}
	
	if (seconds==60) {
		minutes++;
		seconds=0;

		if (minutes == 60) {
			hours++;
			minutes=0;
			
			if (hours == 24) {
				hours = 0;
			}
		}
		
		timeStr[0]=hours/10+'0';
		timeStr[1]=hours%10+'0';
		timeStr[2]=':';
		timeStr[3]=minutes/10+'0';
		timeStr[4]=minutes%10+'0';
		timeStr[5]=' ';
		
		//sprintf_P(timeStr,PSTR("%02d:%02d - liebe dich!"),hours, minutes);
	}
	


	sei();
	
}


// Handles Keypressed in Menu
void handleMenu(void) {
	unsigned char inMenu=0;
	unsigned char key = 0;
	unsigned char sel=STD;		
	
	while (inMenu == 0) {
	
		key = 0;
		
		while (bit_is_clear(PIND, PD2)) {
			key = DOWN;
		}
		while (bit_is_clear(PIND, PD3)) {
			key = UP;
		}
		
		if (key == DOWN || key == UP) {

			if (menu == ENTER) {
				
				if (key == UP) {

					menu = sel;
					if (menu == STD) sprintf (menuStr, "%02d",hours);
					else sprintf (menuStr,"%02d",minutes);
						
				} else if (key == DOWN) {
				
					if (sel == MIN) {

						sel = STD;
						sprintf (menuStr, "ms");
					
					} else if (sel == STD) {
					
						sel = MIN;
						sprintf (menuStr, "mm");

					}
					
				}
			} else if (menu == STD) {
				if (key == UP) {
					inMenu=1;
				} else if (key == DOWN) {
					hours++;
					if (hours == 24) hours=0;
					sprintf (menuStr, "%02d",hours);
				}
			} else if (menu == MIN) {
				if (key == UP) {
					inMenu=1;
				} else if (key == DOWN) {
					minutes++;
					if (minutes == 60) minutes=0;
					sprintf (menuStr,"%02d",minutes);
				}
				
			}
			
		}		
	}
	
	menu = OFF;
}

// ISR for UP Key Pressed
INTERRUPT(SIG_INTERRUPT1)
{
	TCCR2 = 0x00;	// Disable some Interrupts
	GICR &= 0x7F;
	
	delay(10);
	if (bit_is_clear(PIND, PD3)) {

		menu = ENTER;
	
		scrollpos=0;
		poscnt = 0;
		
		sprintf (menuStr, "ms");
	
		while (bit_is_clear(PIND, 3));

		handleMenu();
	}

	timeStr[0]=hours/10+'0';
	timeStr[1]=hours%10+'0';
	timeStr[2]=':';
	timeStr[3]=minutes/10+'0';
	timeStr[4]=minutes%10+'0';
	timeStr[5]=' ';
	
	TCCR2 = 0x05;
	GICR |= 0x80;
}

int main(void) {
	// Initialize Ports
	DDRC = 0x07;
	DDRD = 0xF0;
	DDRB = 0x1F;
	
	PORTC = 0x00;
	PORTD = 0x0F;
	PORTB = 0x00;
	
	MCUCR = 0x0A;	// 0x0A Int0+Int1 on low level
	GICR |= 0x80;	// Enable Int0+Int1
	
	TCCR0 = SCANPRES;
	TCNT0 = SCANFREQ;
	TIMSK |= 0x01;
	
	ASSR = 0x08;
	TCNT2 = 0x00;
	TCCR2 = 0x05; // 5
	while (ASSR&0x07);
	
	TIMSK |= 0x40;
	sprintf (menuStr,"mh");
	timeStr[0]=hours/10+'0';
	timeStr[1]=hours%10+'0';
	timeStr[2]=':';
	timeStr[3]=minutes/10+'0';
	timeStr[4]=minutes%10+'0';
	timeStr[5]=' ';

	//sprintf_P(timeStr,PSTR("%02d:%02d - liebe dich!"),hours, minutes);
	strcatP(timeStr,PSTR("hallo meli, liebe dich! ;-)"),6);

	
	clearBuf();
	
	sei();
	

	while (1);
	
}
